<?php

namespace Tests\Feature;

use App\Models\ActivityLog;
use App\Models\Family;
use App\Models\Invitation;
use App\Models\User;
use App\Models\FamilyWallPost;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Str;
use Tests\TestCase;
use Database\Seeders\RolesAndPermissionsSeeder;

class ActivityLoggingTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        $this->seed(RolesAndPermissionsSeeder::class);
    }

    private function actingAsAdmin(): User
    {
        $user = User::factory()->create();
        $user->assignRole('admin');

        return $user;
    }

    public function test_invitation_creation_logs_activity(): void
    {
        $user = $this->actingAsAdmin();
        $family = Family::factory()->create(['created_by' => $user->id]);

        $this->actingAs($user, 'sanctum');

        $response = $this->postJson("/api/families/{$family->id}/invitations", [
            'email' => 'invitee@example.com',
        ]);

        $response->assertCreated();
        $invitationId = $response->json('id');

        $this->assertDatabaseHas('activity_logs', [
            'event' => 'invitation.created',
            'subject_type' => Invitation::class,
            'subject_id' => $invitationId,
            'family_id' => $family->id,
        ]);
    }

    public function test_invitation_acceptance_logs_activity(): void
    {
        $admin = $this->actingAsAdmin();
        $family = Family::factory()->create(['created_by' => $admin->id]);

        $invitation = $family->invitations()->create([
            'inviter_user_id' => $admin->id,
            'email' => 'invitee@example.com',
            'token' => Str::uuid()->toString(),
            'status' => 'pending',
            'expires_at' => now()->addDay(),
        ]);

        $response = $this->postJson('/api/invitations/accept', [
            'token' => $invitation->token,
        ]);

        $response->assertOk();

        $this->assertDatabaseHas('activity_logs', [
            'event' => 'invitation.accepted',
            'subject_type' => Invitation::class,
            'subject_id' => $invitation->id,
            'family_id' => $family->id,
        ]);
    }

    public function test_wall_post_creation_logs_activity(): void
    {
        $user = $this->actingAsAdmin();
        $family = Family::factory()->create(['created_by' => $user->id]);

        $this->actingAs($user, 'sanctum');

        $response = $this->postJson("/api/families/{$family->id}/wall-posts", [
            'content' => 'Welcome to the family wall!'
        ]);

        $response->assertCreated();
        $postId = $response->json('id');

        $this->assertDatabaseHas('activity_logs', [
            'event' => 'wall_post.created',
            'subject_type' => \App\Models\FamilyWallPost::class,
            'subject_id' => $postId,
            'family_id' => $family->id,
        ]);
    }

    public function test_wall_post_comment_logs_activity(): void
    {
        $user = $this->actingAsAdmin();
        $family = Family::factory()->create(['created_by' => $user->id]);
        $post = FamilyWallPost::factory()->create([
            'family_id' => $family->id,
            'author_id' => $user->id,
        ]);

        $this->actingAs($user, 'sanctum');

        $response = $this->postJson("/api/families/{$family->id}/wall-posts/{$post->id}/comments", [
            'content' => 'Great update!'
        ]);

        $response->assertCreated();
        $commentId = $response->json('id');

        $this->assertDatabaseHas('activity_logs', [
            'event' => 'wall_post.comment_created',
            'subject_type' => \App\Models\FamilyWallPostComment::class,
            'subject_id' => $commentId,
            'family_id' => $family->id,
        ]);
    }

    public function test_wall_post_reaction_logs_activity(): void
    {
        $user = $this->actingAsAdmin();
        $family = Family::factory()->create(['created_by' => $user->id]);
        $post = FamilyWallPost::factory()->create([
            'family_id' => $family->id,
            'author_id' => $user->id,
        ]);

        $this->actingAs($user, 'sanctum');

        $response = $this->postJson("/api/families/{$family->id}/wall-posts/{$post->id}/reactions", [
            'type' => 'love',
        ]);

        $response->assertStatus(201);

        $this->assertDatabaseHas('activity_logs', [
            'event' => 'wall_post.reacted',
            'subject_type' => \App\Models\FamilyWallPostReaction::class,
            'subject_id' => $post->reactions()->where('user_id', $user->id)->first()->id,
            'family_id' => $family->id,
        ]);
    }
}
