<?php

namespace App\Services;

use App\Jobs\SendInvitationNotifications;
use App\Models\Family;
use App\Models\Invitation;
use App\Models\User;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Support\Carbon;
use Illuminate\Support\Str;

class InvitationService
{
    public function listForFamily(Family $family, int $perPage = 15): LengthAwarePaginator
    {
        return $family->invitations()
            ->latest()
            ->paginate($perPage);
    }

    public function createInvitation(Family $family, array $data, User $inviter): Invitation
    {
        $token = $data['token'] ?? Str::uuid()->toString();
        $expiresAt = $data['expires_at'] ?? Carbon::now()->addDays(7);

        $invitation = $family->invitations()->create([
            'inviter_user_id' => $inviter->id,
            'email' => $data['email'] ?? null,
            'phone' => $data['phone'] ?? null,
            'token' => $token,
            'status' => 'pending',
            'expires_at' => $expiresAt,
            'accepted_person_id' => $data['accepted_person_id'] ?? null,
        ]);

        SendInvitationNotifications::dispatch($invitation->id);

        return $invitation;
    }

    public function revokeInvitation(Invitation $invitation): Invitation
    {
        if ($invitation->status === 'pending') {
            $invitation->update(['status' => 'revoked']);
        }

        return $invitation->fresh();
    }

    public function acceptInvitation(Invitation $invitation, ?int $acceptedPersonId = null): Invitation
    {
        if ($invitation->status === 'pending') {
            $invitation->update([
                'status' => 'accepted',
                'accepted_person_id' => $acceptedPersonId,
            ]);
        }

        return $invitation->fresh();
    }
}
