<?php

namespace App\Notifications;

use App\Models\Family;
use App\Models\FamilyAnnouncement;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class FamilyAnnouncementNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(public Family $family, public FamilyAnnouncement $announcement)
    {
        $this->onQueue('notifications');
    }

    public function via(object $notifiable): array
    {
        $channels = ['database'];
        if (method_exists($notifiable, 'getEmailForPasswordReset') || !empty($notifiable->email)) {
            $channels[] = 'mail';
        }
        return $channels;
    }

    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('New family announcement: ' . $this->announcement->title)
            ->greeting('Hello ' . ($notifiable->name ?? ''))
            ->line('A new announcement was posted in ' . $this->family->name . '.')
            ->line('Title: ' . $this->announcement->title)
            ->line($this->announcement->content)
            ->action('View Announcements', route('families.announcements.index', $this->family->id))
            ->line('— FamilyTree');
    }

    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'family_announcement',
            'family_id' => $this->family->id,
            'family_name' => $this->family->name,
            'title' => $this->announcement->title,
            'content_preview' => mb_substr($this->announcement->content, 0, 140),
        ];
    }
}
