<?php

namespace App\Http\Controllers;

use App\Jobs\ProcessPaystackWebhook;
use App\Models\IntegrationEvent;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class WebhookController extends Controller
{
    public function handle(Request $request, string $provider)
    {
        $payload = $request->all();
        $signature = $request->header('x-paystack-signature');

        if ($provider === 'paystack') {
            $this->ensurePaystackSignature($request, $signature);
        }

        $event = IntegrationEvent::create([
            'provider' => $provider,
            'event' => $payload['event'] ?? null,
            'payload' => $payload,
            'status' => 'pending',
        ]);

        if ($provider === 'paystack') {
            ProcessPaystackWebhook::dispatch($event);
        }

        return response()->json(['status' => 'accepted']);
    }

    private function ensurePaystackSignature(Request $request, ?string $signature): void
    {
        $secret = config('services.paystack.webhook_secret', env('PAYSTACK_WEBHOOK_SECRET'));

        if (!$secret) {
            Log::warning('Paystack webhook secret not configured.');
            return;
        }

        $computed = hash_hmac('sha512', $request->getContent(), $secret);

        if (!hash_equals($computed, (string) $signature)) {
            abort(403, 'Invalid Paystack signature.');
        }
    }
}
