<?php

namespace App\Http\Controllers;

use App\Models\ActivityLog;
use App\Models\Plan;
use App\Models\Subscription;
use App\Models\SubscriptionInvoice;
use Carbon\Carbon;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;

class SubscriptionOverviewController extends Controller
{
    public function index(Request $request)
    {
        $summary = $this->buildSummaryMetrics();
        $planPerformance = $this->planPerformance();
        $subscriptionTrend = $this->subscriptionTrend();
        $revenueTrend = $this->revenueTrend();
        $recentInvoices = $this->recentInvoices();
        $recentEvents = $this->recentSubscriptionEvents();
        $subscriptions = $this->latestSubscriptions($request);

        return view('subscriptions.index', compact(
            'summary',
            'planPerformance',
            'subscriptionTrend',
            'revenueTrend',
            'recentInvoices',
            'recentEvents',
            'subscriptions'
        ));
    }

    private function buildSummaryMetrics(): array
    {
        $activeStatuses = ['active', 'trialing', 'pending'];
        $activeCount = Subscription::query()->whereIn('status', $activeStatuses)->count();
        $trialCount = Subscription::query()->where('status', 'trialing')->count();
        $canceledCount = Subscription::query()->where('status', 'canceled')->count();

        $mrrCents = Subscription::query()
            ->whereIn('status', ['active', 'trialing'])
            ->with('plan:id,price_cents')
            ->get()
            ->sum(fn (Subscription $subscription) => $subscription->plan?->price_cents ?? 0);

        $lastMonthRevenue = SubscriptionInvoice::query()
            ->where('status', 'paid')
            ->where('created_at', '>=', Carbon::now()->subMonth())
            ->sum('amount_cents');

        return [
            [
                'label' => 'Active Subscriptions',
                'value' => number_format($activeCount),
                'delta' => '+'.number_format($activeCount).' total',
                'icon' => 'autorenew',
            ],
            [
                'label' => 'In Trial',
                'value' => number_format($trialCount),
                'delta' => 'Onboarding families',
                'icon' => 'hourglass_top',
            ],
            [
                'label' => 'Canceled Accounts',
                'value' => number_format($canceledCount),
                'delta' => 'Lifetime cancellations',
                'icon' => 'block',
            ],
            [
                'label' => 'Monthly Recurring Revenue',
                'value' => '$'.number_format($mrrCents / 100, 2),
                'delta' => '$'.number_format($lastMonthRevenue / 100, 2).' collected last 30d',
                'icon' => 'stacked_line_chart',
            ],
        ];
    }

    private function planPerformance(): Collection
    {
        return Plan::query()
            ->where('is_active', true)
            ->withCount(['subscriptions as active_subscriptions_count' => function ($query) {
                $query->whereIn('status', ['active', 'trialing']);
            }])
            ->get()
            ->map(function (Plan $plan) {
                $revenue = $plan->active_subscriptions_count * ($plan->price_cents / 100);

                return [
                    'name' => $plan->name,
                    'interval' => ucfirst($plan->interval),
                    'price' => '$'.number_format($plan->price_cents / 100, 2),
                    'active_subscriptions' => number_format($plan->active_subscriptions_count),
                    'estimated_revenue' => '$'.number_format($revenue, 2),
                ];
            });
    }

    private function subscriptionTrend(int $months = 6): Collection
    {
        $start = Carbon::now()->startOfMonth()->subMonths($months - 1);

        return collect(range(0, $months - 1))->map(function (int $offset) use ($start) {
            $periodStart = (clone $start)->addMonths($offset);
            $periodEnd = (clone $periodStart)->endOfMonth();

            $count = Subscription::query()
                ->whereBetween('created_at', [$periodStart, $periodEnd])
                ->count();

            return [
                'label' => $periodStart->format('M Y'),
                'value' => $count,
            ];
        });
    }

    private function revenueTrend(int $months = 6): Collection
    {
        $start = Carbon::now()->startOfMonth()->subMonths($months - 1);

        return collect(range(0, $months - 1))->map(function (int $offset) use ($start) {
            $periodStart = (clone $start)->addMonths($offset);
            $periodEnd = (clone $periodStart)->endOfMonth();

            $amount = SubscriptionInvoice::query()
                ->where('status', 'paid')
                ->whereBetween('created_at', [$periodStart, $periodEnd])
                ->sum('amount_cents');

            return [
                'label' => $periodStart->format('M Y'),
                'value' => round($amount / 100, 2),
            ];
        });
    }

    private function recentInvoices(int $limit = 8): Collection
    {
        return SubscriptionInvoice::query()
            ->with(['subscription.plan'])
            ->latest('created_at')
            ->take($limit)
            ->get()
            ->map(function (SubscriptionInvoice $invoice) {
                $occurredAt = $invoice->paid_at ?? $invoice->created_at;

                return [
                    'id' => $invoice->id,
                    'plan' => $invoice->subscription?->plan?->name ?? 'Unknown Plan',
                    'amount' => '$'.number_format($invoice->amount_cents / 100, 2),
                    'status' => ucfirst($invoice->status),
                    'timestamp' => optional($occurredAt)->diffForHumans() ?? '—',
                ];
            });
    }

    private function recentSubscriptionEvents(int $limit = 6): Collection
    {
        return ActivityLog::query()
            ->where('event', 'like', 'subscription.%')
            ->latest('created_at')
            ->take($limit)
            ->get()
            ->map(function (ActivityLog $log) {
                return [
                    'event' => ucwords(str_replace(['subscription.', '_'], ['', ' '], $log->event)),
                    'actor' => $log->user?->name ?? 'System',
                    'time' => optional($log->created_at)->diffForHumans(),
                ];
            });
    }

    private function latestSubscriptions(Request $request): LengthAwarePaginator
    {
        $perPage = (int) $request->integer('per_page', 10);
        $perPage = $perPage > 0 ? min($perPage, 50) : 10;

        return Subscription::query()
            ->with(['plan', 'subscriber'])
            ->latest('created_at')
            ->paginate($perPage);
    }
}
