<?php

namespace App\Http\Controllers;

use App\Models\Person;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class PersonController extends Controller
{
    public function index(Request $request)
    {
        $people = Person::query()->latest()->paginate(15);
        return response()->json($people);
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'user_id' => ['nullable', 'integer', 'exists:users,id'],
            'first_name' => ['required', 'string', 'max:255'],
            'last_name' => ['required', 'string', 'max:255'],
            'middle_name' => ['nullable', 'string', 'max:255'],
            'gender' => ['nullable', Rule::in(['male','female','other'])],
            'birth_date' => ['nullable', 'date'],
            'death_date' => ['nullable', 'date', 'after_or_equal:birth_date'],
            'place_of_birth' => ['nullable', 'string', 'max:255'],
            'bio' => ['nullable', 'string'],
            'photo_url' => ['nullable', 'url'],
            'meta' => ['nullable', 'array'],
            'is_memorialized' => ['nullable', 'boolean'],
            'memorialized_at' => ['nullable', 'date'],
            'memorialized_by' => ['nullable', 'integer', 'exists:users,id'],
            'memorial_note' => ['nullable', 'string'],
        ]);

        $person = Person::create($data);

        return response()->json($person, 201);
    }

    public function show(Person $person)
    {
        return response()->json($person);
    }

    public function update(Request $request, Person $person)
    {
        $data = $request->validate([
            'user_id' => ['sometimes', 'integer', 'exists:users,id'],
            'first_name' => ['sometimes', 'string', 'max:255'],
            'last_name' => ['sometimes', 'string', 'max:255'],
            'middle_name' => ['nullable', 'string', 'max:255'],
            'gender' => ['nullable', Rule::in(['male','female','other'])],
            'birth_date' => ['nullable', 'date'],
            'death_date' => ['nullable', 'date', 'after_or_equal:birth_date'],
            'place_of_birth' => ['nullable', 'string', 'max:255'],
            'bio' => ['nullable', 'string'],
            'photo_url' => ['nullable', 'url'],
            'meta' => ['nullable', 'array'],
            'is_memorialized' => ['nullable', 'boolean'],
            'memorialized_at' => ['nullable', 'date'],
            'memorialized_by' => ['nullable', 'integer', 'exists:users,id'],
            'memorial_note' => ['nullable', 'string'],
        ]);

        $person->update($data);

        return response()->json($person);
    }

    public function memorialize(Request $request, Person $person)
    {
        $data = $request->validate([
            'is_memorialized' => ['required', 'boolean'],
            'memorial_note' => ['nullable', 'string'],
        ]);

        abort_unless($request->user()?->can('people.manage'), 403);

        if ($data['is_memorialized']) {
            $person->update([
                'is_memorialized' => true,
                'memorialized_at' => now(),
                'memorialized_by' => $request->user()?->id,
                'memorial_note' => $data['memorial_note'] ?? null,
            ]);
        } else {
            $person->update([
                'is_memorialized' => false,
                'memorialized_at' => null,
                'memorialized_by' => null,
                'memorial_note' => null,
            ]);
        }

        return response()->json($person->fresh());
    }

    public function destroy(Person $person)
    {
        $person->delete();
        return response()->json(['message' => 'Deleted']);
    }
}
