<?php

namespace App\Http\Controllers;

use App\Models\Family;
use App\Models\FamilyWallPost;
use App\Models\FamilyWallPostComment;
use App\Services\ActivityLogger;
use Illuminate\Http\Request;

class FamilyWallPostCommentController extends Controller
{
    public function __construct(private ActivityLogger $activityLogger)
    {
    }

    public function index(Request $request, Family $family, FamilyWallPost $wallPost)
    {
        $this->authorizeView($request, $family, $wallPost);

        $perPage = (int) $request->integer('per_page', 15);
        $perPage = $perPage > 0 ? min($perPage, 100) : 15;

        $comments = $wallPost->comments()->with('author')->latest()->paginate($perPage);

        return response()->json($comments);
    }

    public function store(Request $request, Family $family, FamilyWallPost $wallPost)
    {
        $this->authorizeComment($request, $family, $wallPost);

        $data = $request->validate([
            'content' => ['required', 'string'],
            'meta' => ['nullable', 'array'],
        ]);

        $comment = $wallPost->comments()->create([
            'user_id' => $request->user()->id,
            'content' => $data['content'],
            'meta' => $data['meta'] ?? null,
        ]);

        $this->activityLogger->log('wall_post.comment_created', $comment, [
            'family_id' => $family->id,
            'wall_post_id' => $wallPost->id,
        ], $request, $family->id);

        return response()->json($comment->load('author'), 201);
    }

    public function update(Request $request, Family $family, FamilyWallPost $wallPost, FamilyWallPostComment $comment)
    {
        $this->authorizeEdit($request, $family, $wallPost, $comment);

        $data = $request->validate([
            'content' => ['required', 'string'],
            'meta' => ['nullable', 'array'],
        ]);

        $comment->update($data);

        $this->activityLogger->log('wall_post.comment_updated', $comment, [
            'family_id' => $family->id,
            'wall_post_id' => $wallPost->id,
        ], $request, $family->id);

        return response()->json($comment->fresh('author'));
    }

    public function destroy(Request $request, Family $family, FamilyWallPost $wallPost, FamilyWallPostComment $comment)
    {
        $this->authorizeEdit($request, $family, $wallPost, $comment);

        $comment->delete();

        $this->activityLogger->log('wall_post.comment_deleted', $comment, [
            'family_id' => $family->id,
            'wall_post_id' => $wallPost->id,
        ], $request, $family->id);

        return response()->json(['message' => 'Comment removed']);
    }

    private function authorizeView(Request $request, Family $family, FamilyWallPost $wallPost): void
    {
        abort_unless($wallPost->family_id === $family->id, 404);

        $user = $request->user();
        $isMember = $family->members()->where('person_id', optional($user->person)->id)->exists();
        abort_unless($user?->can('families.manage') || $isMember, 403);
    }

    private function authorizeComment(Request $request, Family $family, FamilyWallPost $wallPost): void
    {
        abort_unless($wallPost->family_id === $family->id, 404);

        $user = $request->user();
        $canManage = $user?->can('families.manage');
        $isMember = $family->members()->where('person_id', optional($user->person)->id)->exists();

        abort_unless($canManage || $isMember, 403);
    }

    private function authorizeEdit(Request $request, Family $family, FamilyWallPost $wallPost, FamilyWallPostComment $comment): void
    {
        abort_unless($wallPost->family_id === $family->id, 404);
        abort_unless($comment->family_wall_post_id === $wallPost->id, 404);

        $user = $request->user();
        $canManage = $user?->can('families.manage');
        $isAuthor = $comment->user_id === $user?->id;

        abort_unless($canManage || $isAuthor, 403);
    }
}
