<?php

namespace App\Http\Controllers;

use App\Models\Family;
use App\Models\SubscriptionPaymentMethod;
use App\Services\SubscriptionPaymentMethodService;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class FamilyPaymentMethodController extends Controller
{
    public function __construct(private SubscriptionPaymentMethodService $paymentMethods)
    {
    }

    public function index(Request $request, Family $family)
    {
        $this->authorize('viewSubscription', $family);

        $perPage = (int) $request->integer('per_page', 15);
        $perPage = $perPage > 0 ? min($perPage, 100) : 15;

        $methods = $this->paymentMethods->list($family, $perPage);

        return response()->json($methods);
    }

    public function store(Request $request, Family $family)
    {
        $this->authorize('manageSubscription', $family);

        $data = $request->validate([
            'driver' => ['nullable', 'string'],
            'email' => ['nullable', 'email'],
            'authorization_code' => ['nullable', 'string'],
            'card_brand' => ['nullable', 'string'],
            'card_last_four' => ['nullable', 'string', 'size:4'],
            'card_exp_month' => ['nullable', 'integer', 'between:1,12'],
            'card_exp_year' => ['nullable', 'integer', 'min:'.now()->year],
            'signature' => ['nullable', 'string'],
            'metadata' => ['nullable', 'array'],
            'set_default' => ['nullable', 'boolean'],
        ]);

        $method = $this->paymentMethods->store($family, $data, $request);

        return response()->json($method, 201);
    }

    public function setDefault(Request $request, Family $family, SubscriptionPaymentMethod $paymentMethod)
    {
        $this->authorize('manageSubscription', $family);

        abort_unless($paymentMethod->subscriber_id === $family->id && $paymentMethod->subscriber_type === Family::class, 404);

        $updated = $this->paymentMethods->setDefault($family, $paymentMethod, $request);

        return response()->json($updated);
    }

    public function destroy(Request $request, Family $family, SubscriptionPaymentMethod $paymentMethod)
    {
        $this->authorize('manageSubscription', $family);

        abort_unless($paymentMethod->subscriber_id === $family->id && $paymentMethod->subscriber_type === Family::class, 404);

        $this->paymentMethods->delete($family, $paymentMethod, $request);

        return response()->json(['message' => 'Payment method removed']);
    }
}
