<?php

namespace App\Http\Controllers;

use App\Models\Family;
use App\Models\FamilyEvent;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class FamilyEventController extends Controller
{
    public function index(Request $request, Family $family)
    {
        $this->authorizeMember($request, $family);

        $events = FamilyEvent::query()
            ->where('family_id', $family->id)
            ->orderBy('starts_at')
            ->get();

        $canManage = $this->canManage($request, $family);
        $canCreate = $this->canCreate($request, $family);
        $members = $family->members()
            ->select('people.id', 'first_name', 'last_name')
            ->orderBy('first_name')
            ->get()
            ->map(fn($p) => [
                'id' => $p->id,
                'name' => trim(($p->first_name ?? '') . ' ' . ($p->last_name ?? '')) ?: 'Member',
            ]);
        $creatorIds = $family->members()->wherePivot('can_create_events', true)->pluck('people.id')->map(fn($id) => (int) $id)->all();

        $settings = $family->settings ?? [];
        $reminderDays = (array) ($settings['event_reminder_days'] ?? [3]);
        $reminderTime = (string) ($settings['event_reminder_time'] ?? '07:00');
        $prefill = null;
        $editId = (int) $request->query('edit', 0);
        if ($editId > 0) {
            $e = FamilyEvent::where('family_id', $family->id)->where('id', $editId)->first();
            if ($e) {
                $prefill = [
                    'id' => $e->id,
                    'title' => $e->title,
                    'description' => $e->description,
                    'starts_at' => optional($e->starts_at)->format('Y-m-d\TH:i'),
                    'ends_at' => optional($e->ends_at)->format('Y-m-d\TH:i'),
                    'location' => $e->location,
                ];
            }
        }

        return view('families.events.index', [
            'family' => $family,
            'events' => $events,
            'canManage' => $canManage,
            'canCreate' => $canCreate,
            'members' => $members,
            'creatorIds' => $creatorIds,
            'reminderDays' => $reminderDays,
            'reminderTime' => $reminderTime,
            'prefill' => $prefill,
        ]);
    }

    public function store(Request $request, Family $family)
    {
        $this->authorizeCreate($request, $family);

        $data = $request->validate([
            'title' => ['required','string','max:255'],
            'description' => ['nullable','string'],
            'starts_at' => ['required','date'],
            'ends_at' => ['nullable','date','after_or_equal:starts_at'],
            'location' => ['nullable','string','max:255'],
        ]);

        FamilyEvent::create([
            'family_id' => $family->id,
            'title' => $data['title'],
            'description' => $data['description'] ?? null,
            'starts_at' => $data['starts_at'],
            'ends_at' => $data['ends_at'] ?? null,
            'location' => $data['location'] ?? null,
        ]);

        return redirect()->route('families.events.index', $family)->with('status', 'Event added.');
    }

    public function update(Request $request, Family $family, FamilyEvent $event)
    {
        $this->authorizeManage($request, $family);
        abort_unless($event->family_id === $family->id, 404);

        $data = $request->validate([
            'title' => ['required','string','max:255'],
            'description' => ['nullable','string'],
            'starts_at' => ['required','date'],
            'ends_at' => ['nullable','date','after_or_equal:starts_at'],
            'location' => ['nullable','string','max:255'],
        ]);

        $event->title = $data['title'];
        $event->description = $data['description'] ?? null;
        $event->starts_at = $data['starts_at'];
        $event->ends_at = $data['ends_at'] ?? null;
        $event->location = $data['location'] ?? null;
        $event->save();

        return redirect()->route('families.events.index', $family)->with('status', 'Event updated.');
    }

    public function destroy(Request $request, Family $family, FamilyEvent $event)
    {
        $this->authorizeManage($request, $family);
        abort_unless($event->family_id === $family->id, 404);
        $event->delete();
        return redirect()->route('families.events.index', $family)->with('status', 'Event deleted.');
    }

    private function authorizeMember(Request $request, Family $family): void
    {
        $user = $request->user();
        $isMember = $user && $family->members()->where('people.user_id', $user->id)->exists();
        abort_unless($isMember, 403);
    }

    private function authorizeManage(Request $request, Family $family): void
    {
        $user = $request->user();
        $isOwner = $user && $family->members()
            ->wherePivot('role', 'owner')
            ->where('people.user_id', $user->id)
            ->exists();
        abort_unless($user && ($user->can('families.manage') || $isOwner), 403);
    }

    private function canCreate(Request $request, Family $family): bool
    {
        $user = $request->user();
        if (! $user) return false;
        if ($this->canManage($request, $family)) return true;
        return $family->members()
            ->where('people.user_id', $user->id)
            ->wherePivot('can_create_events', true)
            ->exists();
    }

    private function authorizeCreate(Request $request, Family $family): void
    {
        abort_unless($this->canCreate($request, $family), 403);
    }

    private function canManage(Request $request, Family $family): bool
    {
        $user = $request->user();
        $isOwner = $user && $family->members()
            ->wherePivot('role', 'owner')
            ->where('people.user_id', $user->id)
            ->exists();
        return $user && ($user->can('families.manage') || $isOwner);
    }

    public function updatePermissions(Request $request, Family $family)
    {
        $this->authorizeManage($request, $family);
        $data = $request->validate([
            'creators' => ['array'],
            'creators.*' => ['integer'],
        ]);

        $allowedIds = $family->members()->pluck('people.id')->map(fn ($id) => (int) $id)->all();
        $selected = collect($data['creators'] ?? [])
            ->map(fn ($id) => (int) $id)
            ->filter(fn ($id) => in_array($id, $allowedIds, true))
            ->unique()
            ->values()
            ->all();

        foreach ($allowedIds as $pid) {
            $family->members()->updateExistingPivot($pid, [
                'can_create_events' => in_array($pid, $selected, true),
            ]);
        }

        return redirect()->route('families.events.index', $family)->with('status', 'Event permissions updated.');
    }

    public function updateReminderSettings(Request $request, Family $family)
    {
        $this->authorizeManage($request, $family);
        $data = $request->validate([
            'reminder_days' => ['nullable'],
            'reminder_time' => ['required', 'date_format:H:i'],
        ]);

        $daysInput = $request->input('reminder_days');
        if (is_string($daysInput)) {
            $days = array_filter(array_map('trim', explode(',', $daysInput)), fn ($v) => $v !== '');
        } elseif (is_array($daysInput)) {
            $days = $daysInput;
        } else {
            $days = [];
        }

        $days = collect($days)
            ->map(fn ($d) => (int) $d)
            ->filter(fn ($d) => $d >= 0 && $d <= 60)
            ->unique()
            ->values()
            ->all();

        $settings = $family->settings ?? [];
        $settings['event_reminder_days'] = $days;
        $settings['event_reminder_time'] = $data['reminder_time'];
        $family->settings = $settings;
        $family->save();

        return redirect()->route('families.events.index', $family)->with('status', 'Reminder settings updated.');
    }
}
