<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\TrafficEvent;
use App\Models\TrafficVisit;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class TrafficAnalyticsController extends Controller
{
    public function index(Request $request)
    {
        $from = $request->date('from') ? Carbon::parse($request->get('from'))->startOfDay() : Carbon::now()->subDays(7)->startOfDay();
        $to = $request->date('to') ? Carbon::parse($request->get('to'))->endOfDay() : Carbon::now()->endOfDay();

        $visitsQuery = TrafficVisit::query()->whereBetween('created_at', [$from, $to]);
        $eventsQuery = TrafficEvent::query()->whereBetween('created_at', [$from, $to]);

        $totals = [
            'visits' => (clone $visitsQuery)->count(),
            'unique_sessions' => (clone $visitsQuery)->distinct('session_id')->count('session_id'),
            'unique_users' => (clone $visitsQuery)->whereNotNull('user_id')->distinct('user_id')->count('user_id'),
        ];

        $daily = (clone $visitsQuery)
            ->selectRaw('DATE(created_at) as day, COUNT(*) as total')
            ->groupBy('day')
            ->orderBy('day')
            ->get();

        $topPages = (clone $visitsQuery)
            ->select('path', DB::raw('COUNT(*) as total'))
            ->groupBy('path')
            ->orderByDesc('total')
            ->limit(10)
            ->get();

        $topCountries = (clone $visitsQuery)
            ->whereNotNull('country')
            ->select('country', DB::raw('COUNT(*) as total'))
            ->groupBy('country')
            ->orderByDesc('total')
            ->limit(10)
            ->get();

        $devices = (clone $visitsQuery)
            ->select('device', DB::raw('COUNT(*) as total'))
            ->groupBy('device')
            ->orderByDesc('total')
            ->get();

        $browsers = (clone $visitsQuery)
            ->select('browser', DB::raw('COUNT(*) as total'))
            ->groupBy('browser')
            ->orderByDesc('total')
            ->get();

        $platforms = (clone $visitsQuery)
            ->select('platform', DB::raw('COUNT(*) as total'))
            ->groupBy('platform')
            ->orderByDesc('total')
            ->get();

        $recentVisits = (clone $visitsQuery)->with('user:id,name')->latest('created_at')->paginate(25)->withQueryString();
        $recentEvents = (clone $eventsQuery)->with('user:id,name')->latest('created_at')->paginate(25)->withQueryString();

        return view('admin.analytics.index', [
            'from' => $from,
            'to' => $to,
            'totals' => $totals,
            'daily' => $daily,
            'topPages' => $topPages,
            'topCountries' => $topCountries,
            'devices' => $devices,
            'browsers' => $browsers,
            'platforms' => $platforms,
            'recentVisits' => $recentVisits,
            'recentEvents' => $recentEvents,
        ]);
    }
}
